/************************************************************************
* \file: trace_dlt.c
*
* \version: $Id: trace_dlt.c,v 1.4 2012/05/28 10:29:00 jayanth.mc Exp $
*
* This file implements the Trace back-end for DLT.
*
* \component: Gen2 Trace
*
* \author Mikhail Durnev mikhail_durnev@mentor.com
* \copyright: (c) 2012 ADIT
*
***********************************************************************/
#include <endian.h> 
#include <dlt/dlt.h>
#include "adit_dlt.h"
#include "trace_base.h"
#include "trace_dlt_utility.h"
#include "trace_be_common.h"
#include "trace_impl.h"
#include "trace_common.h"

#define TRACE_DLT_MAX_TRY 3
DltContext g_dlt_context;
DltContextData g_dlt_contextdata;

/* ptr to shared memory created by Trace FE */
EXPORT TRACE_mgr* g_TRACE_mgr = NULL;

#ifndef TRACE_ALD_INTERFACE_ENABLED
/* semaphore e for process termination */
    IMPORT sem_t g_semExit;
#else
    /* eventfd to signal main thread for process termination */
    #include <sys/eventfd.h>
    IMPORT int event_fd;
#endif

/*Private declarations*/
#define EXIT_SUCCESS 0
#define EXIT_FAILURE 1


/**
 * Write data to DLT
 *
 * \parm mgr  Pointer to Trace manager
 * \parm curq Pointer to current packet
 *
 * return \li E_OK     If successful
 *        \li E_IO     In case of socket errors
 *        \li E_DISWAI In case of shutdown request
 *                     Else error returned by concrete implementation
 */

EXPORT ER TRACE_dlt_wr(TRACE_mgr* mgr, TRACE_q* curq) {
  ER  rc      = E_OK;
  S32 ret     = E_OK;
  S32 wrsz    = 1;
  U8  rawpkt[sizeof(TRACE_TRACEOUT_MSG) + 2];
  U8  txtpkt[(sizeof(TRACE_TRACEOUT_MSG) + 2) * 3 + 1];

  TRACE_TRACEOUT_MSG* pkt         = NULL;
  TRACE_OUT_PKT*      trace_pkt   = NULL;

  pkt         = (TRACE_TRACEOUT_MSG*)(VP) curq->buf;
  trace_pkt   = (VP) &pkt->pkt;

  rawpkt[TRACE_FRAME_CNT] = 0;
  if(pkt->tm_stamp != TRACE_TIMESTAMP_DISABLE)
  {
    rawpkt[TRACE_FRAME_CNT]++;
    TRACE_construct_frame(trace_pkt, rawpkt + wrsz, &wrsz);
  }

  trace_pkt = (TRACE_OUT_PKT*)(VP) &pkt->pkt.trc_msg.out_pkt;
  TRACE_construct_frame(trace_pkt, rawpkt + wrsz, &wrsz);
  rawpkt[TRACE_FRAME_CNT]++;

  if(wrsz > 0)
  {
    static S32 id = 0;
    S32 i;

    /* convert binary data to text */
    for (i = 0; i < wrsz; i++)
    {
        (void)TRACE_bin2hex(rawpkt[i], &txtpkt[i + i + i]);
    }

    rc = E_OK;
    if (dlt_user_log_write_start_id(&g_dlt_context, &g_dlt_contextdata,
                                    DLT_LOG_FATAL, id++) > 0)
    {
        if (dlt_user_log_write_string(&g_dlt_contextdata, (const char*)txtpkt)
             >= 0)
        {
           /* RTC 250640 - Removed the retry mechansim to avoid junk message data to DLT */
           ret = dlt_user_log_write_finish(&g_dlt_contextdata);
           if(ret >= E_OK)
           {
              ret = E_OK;
           }
           else
           {
              TRACE_SYSLOG(TRACE_SYSLOG_ERROR, " dlt_user_log_write_finish() failed unknown error \n");
           }

           if(ret < E_OK)
           {
               /* If dlt-daemon is present, then return of system is 0, else 1 */
               /* Initial checking in below system() was ps | grep dlt-daemon since ps command
                * has removed in system(only ps with option is supported). Now added ps -ax. Just putting syslog here */
               if( (rc = system("ps -ax | grep dlt-daemon | grep -v grep") ) != E_OK) /* dlt-daemon exited */
               {
                  rc = E_OK;
                  TRACE_SYSLOG(TRACE_SYSLOG_ERROR, " dlt-daemon exited with unknown error \n");
               }
               else
               {
                  TRACE_SYSLOG(TRACE_SYSLOG_ERROR, " dlt-daemon is present. dlt_user_log_write_finish() failed with unknown error. discarding the dlt message \n");
               }
           }
        }
        else
        {
          TRACE_SYSLOG(TRACE_SYSLOG_ERROR, " dlt_user_log_write_string() failed with unknown error. Simply discard message \n");
        }
    }
    else if (dlt_user_log_write_start_id(&g_dlt_context, &g_dlt_contextdata,
                                         DLT_LOG_FATAL, id++) == 0)
    {
      /* @uns7kor - this conditional block is used to turn off traces when the log level is set to DLT_LOG_OFF - RTC 240667 fix */
      TRACE_SYSLOG(TRACE_SYSLOG_ERROR, " dlt_user_log_write_start_id() returned 0, for trace level mismatch. Message is discarded \n");
    }
    else
    {
      TRACE_SYSLOG(TRACE_SYSLOG_ERROR, " dlt_user_log_write_start_id()\n");
    }
  }
  mgr = mgr;
  return rc;
}



/**
* Check for Buffer Overwritten case, and send the info to TTFis
*
* \parm mgr        pointer to Trace manager
* \parm fp         pointer to a io func
* \parm ovrcnt     no of bytes overwritten
* return           None
*/
EXPORT void TRACE_snd_ovrcnt_info_dlt(TRACE_mgr* mgr, U32 ovrcnt)
{
    static TRACE_TRACEOUT_MSG pkt;
    TRACE_q     q;
    TRACE_q_sh  q_sh;

    memset(&pkt, 0, sizeof(TRACE_TRACEOUT_MSG));
    memset(&q, 0, sizeof(TRACE_q));
    memset(&q_sh, 0, sizeof(TRACE_q_sh));

    q.sh = &q_sh;
    q.sh->pkt_sz = sizeof(TRACE_TRACEOUT_MSG);
    if((ovrcnt > 0) /*&& (fp != NULL)*/)
    {
        {
            /* create Trace Header */
            TRACE_cre_hdr(sizeof(U32), TR_CLASS_TRACE, TRACE_OVRCNT_INFO,
                &pkt.pkt.trc_msg.out_pkt.hdr, TR_MSG_NORMAL);

            pkt.tm_stamp = TRACE_TIMESTAMP_DISABLE;
            /* fill in Overwrite byte count */
            TRACE_pack_32bit_in_bigendian((ovrcnt * q.sh->pkt_sz),
                (VP)pkt.pkt.trc_msg.out_pkt.payld);
        }
        q.buf = (U8*)(VP)&pkt;
        (void)TRACE_dlt_wr(mgr,&q);
    }
}

/**
* Trace Read Task
*
* Read incomming trace data from physical layer and route to applications.
*
* \param  service_id   Unused
*         data         Incoming data
*         length       Incoming data size in bytes
*
* \return None
*/
LOCAL S32 dlt_user_injection_callback(U32 service_id, VP data, U32 length);
LOCAL S32 dlt_user_injection_callback(U32 service_id, VP data, U32 length)
{
    TRACE_mgr* mgr = g_TRACE_mgr;
    U8  rawpkt[TRACE_MAX_FRAME_SZ] = {0};
    ER  rc                         = E_OK;
    S32 asiz                       = 0;
    S32 indx                       = TRACE_PKTSZ_FIELD;
    S32 stage                      = 0;
    S32 i                          = 0;
    if (length > 0)
    {
        while ((rc >= E_OK) && (stage < 3))
        {
            switch (stage)
            {
                case 0:
                    /* convert the message to binary form */
                    for (i = 0; i < (S32)(length - 2) && 
                                asiz < (S32)TRACE_MAX_FRAME_SZ;
                         i += 3)
                    {
                        S32 bin = TRACE_hex2bin(&((U8*)data)[i]);
                        if (bin < 0 || bin > 0xff)
                        {
                            /* wrong message format */
                            rc = E_FAIL;
                            break;
                        }

                        rawpkt[asiz++] = (U8)bin;
                    }
                break;

                case 1:
                    if((rawpkt[indx] < TRACE_MIN_PKT_SZ) ||
                       (rawpkt[indx] > TRACE_MAX_PKT_SZ))
                    {
                        stage  = 3;
                    }
                break;

                case 2:
                    if(asiz >= rawpkt[indx])
                    {
                        mgr->trc.in_msg = (VP)&rawpkt[indx];

                       (void)TRACE_process_cmd(mgr);
                       if(asiz > rawpkt[indx])
                       {
                           asiz -= rawpkt[indx];
                           indx += rawpkt[indx];
                           stage = 0;
                       }
                    }
                    else
                    {
                        rc = E_FAIL;
                    }
                break;

                default:
                    break;
            }
        stage++;
        }
    }

    service_id = service_id;
    return 0;
}

/**
* Trace shutdown sequence for backend.
*
* \parm mgr Pointer to Trace manager
*
* return \li E_OK    if successful
*                    else error values returned by concrete implementation.
*/
LOCAL ER TRACE_dlt_stop(TRACE_mgr* mgr);
LOCAL ER TRACE_dlt_stop(TRACE_mgr* mgr)
{
    ER             rc     = E_OK;
    U8             cnt    = 0;
    TRACE_cond* cond[]    = {&mgr->ctrl.condWait, &mgr->ctrl.condSleep};

    if(mgr != NULL)
    {
        mgr->prxy.exit = TRUE;

        for (cnt = 0; cnt < TRACE_MAX_TASK; cnt++)
        {
          if(mgr->ctrl.tsk_id[cnt].exist == TRUE)
          {      
            rc = pthread_cancel(mgr->ctrl.tsk_id[cnt].tid);
            mgr->ctrl.tsk_id[cnt].rc = rc;
          }
        }	/*for (cnt = 0; cnt < TRACE_MAX_TASK; cnt++)*/ 

        for (cnt = 0; cnt < TRACE_MAX_TASK; cnt++)
        {
          if( (E_OK == mgr->ctrl.tsk_id[cnt].rc) && (0 != mgr->ctrl.tsk_id[cnt].tid) )
          {
            pthread_join(mgr->ctrl.tsk_id[cnt].tid, NULL);
          }
        }
        /* uninit mutex and condition variable for switch rd wr sync*/
        for (cnt = 0; cnt < UTIL_NO(cond); cnt++)
        {
            TRACE_cond_uninit(cond[cnt]);
        }
    }
    
    dlt_unregister_context(&g_dlt_context);

    dlt_unregister_app();

    dlt_free();

    return rc;
}

/**
* Trace startup sequence
*
* \parm None
*
* return \li E_OK    if successful
*                    else error values returned by concrete implementation.
*/
LOCAL ER TRACE_dlt_init(void);
LOCAL ER TRACE_dlt_init(void)
{
    ER            rc       = E_OK;
    S32           stage    = 0;
    TRACE_EC      ec;
    U32           cnt      = 0;
    TRACE_cond*   cond[]   = {&g_TRACE_mgr->ctrl.condWait, &g_TRACE_mgr->ctrl.condSleep};
    U32    evtptn_master   = TRACE_SIG_WR_SELF;

    struct { TRACE_task task; TRACE_max_task idx; } tsk[] = {
        {TRACE_wr_task,     TRACE_WR_TASK},
        {TRACE_proxy_task,  TRACE_PRXY_TASK},
        {TRACE_chan_task,   TRACE_CHNL_TASK},
        {TRACE_sendcmd_task,  TRACE_APPL_CMD_TASK}
    };

    memset(&ec, 0, sizeof(TRACE_EC));
    if(g_TRACE_mgr != NULL)
    {
        while((stage < 4) && (rc == E_OK))
        {
            switch(stage)
            {
              case 0: 
              rc = sem_wait(&g_TRACE_mgr->sh->sem);
              if(E_OK != rc)
              {
                TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "Gen3: Sem wait of sync Semaphore failed\n");
              }
              else
              {
                /* Read Trace configuration */
                rc = TRACE_rd_cfg(g_TRACE_mgr);
                if(rc != E_OK)
                {
                  TRACE_SYSLOG(TRACE_SYSLOG_NOTICE, "Trace configuration file is not present\n");
                  rc = E_OK;
                }
                rc = sem_post(&g_TRACE_mgr->sh->sem);
                if(E_OK != rc)
                {
                  TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "Gen3: Sem post of sync Semaphore failed\n");
                }
              }
              break;

            case 1: /* Open DLT */
              rc = E_FAIL;
              if (dlt_init() >= 0 && 
                  dlt_register_app(DLT_TRACE_DLT_BE_APID, "TRACE Backend") >= 0 &&
                  dlt_register_context(&g_dlt_context, "CTX", 
                                       "Context for Logging") >= 0 &&
                  dlt_register_injection_callback(&g_dlt_context, 0xFFF, 
                                            dlt_user_injection_callback) >= 0)
              {
                  //dlt_log_init(1); /* Route internal logs to syslog */
                  rc = sem_wait(&g_TRACE_mgr->sh->sem);
                  if(E_OK != rc)
                  {
                    TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "Gen3: Sem wait of sync Semaphore failed\n");
                  }
                  else
                  {
                    rc = sem_post(&g_TRACE_mgr->sh->sem);
                    if(E_OK != rc)
                    {
                      TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "Gen3: Sem post of sync Semaphore failed\n");
                    }
                  }
              }
            break;

            case 2: /* Initialize condition variable and mutexs */
              for(cnt = 0; ( (cnt < UTIL_NO(cond)) && (rc == E_OK) ); cnt++)
              {
                  rc = TRACE_cond_init(cond[cnt]);
              }     
            break;

            case 3: /* Create tasks */
              for(cnt = 0; ( (cnt < UTIL_NO(tsk)) && (rc == E_OK)); cnt++)
              {                
		  rc = TRACE_init_task(tsk[cnt].task, tsk[cnt].idx, g_TRACE_mgr);
              }
              /* start WR tasks*/
              TRACE_set_flag(&g_TRACE_mgr->sh->evt_id[EVT_COMMON], evtptn_master);
            break;

            default:
            break;
            }
            stage++;
        }
        if(rc != E_OK)
        {
            TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "Error happened in trace_dlt_be init()\n");
            (void)TRACE_dlt_stop(g_TRACE_mgr);
        }
        /* PRQA L:L1 */
    }
    return rc;
}

/**
* signal handler
* handles SIGTERM and SIGINT signals
*
* \parm  sig signum
*
* return \li none
*/
LOCAL void TRACE_signal_hdlr_DLT(S32 sig);
LOCAL void TRACE_signal_hdlr_DLT(S32 sig)
{
#ifdef TRACE_ALD_INTERFACE_ENABLED
    uint64_t tmp=1;
#endif
   switch(sig)
   {
    case SIGTERM:
    case SIGINT:
          TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "Caught termination signal in trace_dlt_be, making trace_be active\n");
#ifndef TRACE_ALD_INTERFACE_ENABLED
        /* use only async-signal-safe functions only */        
        sem_post(&g_semExit); 
#else
        if (write(event_fd, &tmp, sizeof(uint64_t))==0)
        {
        }
#endif
    break;

    default:
        /* not handled */
    break;
   }
}


/**
* Entry point for Trace BE
* This routine starts up Trace BE(ttfis) process
* and waits for SIGTERM or SIGINT signals for termination
* Upon receiving termination signal, does clean up and exit
*
* \parm  none
*
* return \li 0  Normal termination
*/
EXPORT ER main()
{
	ER			rc		=	E_OK;
	U32			cnt		=	0;
	S32			sig[]		=	{SIGTERM, SIGINT};
	struct			sigaction sigact;

#ifdef TRACE_ALD_INTERFACE_ENABLED
	int			err_tmp		=	0;
	uint64_t		tmp;
	bool			exit_loop	=	FALSE;
	int			nfds		=	-1;
	int			i		=	0;
#endif
	
	/* Check whether any other instance of trace_dlt_be is already started, if started dont proceed. */
	/* Fix for SWGIII-2024 (SWGIII-1920) MUltiple trace_be should not be started*/
	/*SWGIII-3516:TRACE startup with Invalid configuration fails */
	pid_t pid = TRACE_check_process(TRACEDLT);
	if(pid > 0)
	{
		TRACE_SYSLOG(TRACE_SYSLOG_ERROR,"A trace_dlt_be process is already running with pid:%d\n", pid);
		rc = E_FAIL;

	}
	else
	{
		/* initialise Trace mgr & setup shared memory */
		rc = TRACE_start();
		if(rc >= E_OK)
		{
			/*Demonize the trace_be here itself, to remove all complications*/
			TRACE_daemonize();

			memset(&sigact, 0, sizeof(sigact));
			sigemptyset(&sigact.sa_mask);
			sigact.sa_handler = TRACE_signal_hdlr_DLT;
			sigact.sa_flags   = SA_RESTART;
#ifndef TRACE_ALD_INTERFACE_ENABLED
			/* init semaphore for termintaion signal */
			rc = sem_init(&g_semExit,PTHREAD_PROCESS_PRIVATE,0);
#else

			event_fd = -1;
			//Initailise an eventfd for communication between other TRACE Thread and TRACE main loop
			event_fd = eventfd(0,EFD_NONBLOCK);

			if(event_fd == -1)
			{
				TRACE_SYSLOG(TRACE_SYSLOG_ERROR,"ALD_PLUGIN - Init Failed to create eventfd in trace_be main \n");
				rc = -1;
			}
			else
			{
				rc = TRACE_ALD_interface_init_ald_plugin(&(g_TRACE_mgr->ald_mgr));
				if((g_TRACE_mgr->ald_mgr.epollHandle >= 0) && (rc == 0))
				{
					g_TRACE_mgr->ald_mgr.eventHandle.events = EPOLLIN;
					g_TRACE_mgr->ald_mgr.eventHandle.data.fd = event_fd;

					if(epoll_ctl(g_TRACE_mgr->ald_mgr.epollHandle, EPOLL_CTL_ADD, event_fd, &(g_TRACE_mgr->ald_mgr.eventHandle)) == -1)
					{
						TRACE_SYSLOG(TRACE_SYSLOG_ERROR,"ALD_PLUGIN - Init Failed to create eventfd in trace_be main \n");
						rc = -1;
					}
				}

			}

#endif

			if(rc >= E_OK)
			{
				/* register signal handlers */
				for(cnt = 0; cnt < UTIL_NO(sig); cnt++)
				{
					(void)sigaction(sig[cnt], &sigact, NULL);
				}
				rc = TRACE_dlt_init();

				if(rc >= E_OK)
				{
#ifndef TRACE_ALD_INTERFACE_ENABLED
					/* wait for exit signal */
					sem_wait(&g_semExit);
#else
					//Close up Connection to host in case ALD not yet started.
					if(g_TRACE_mgr->ald_mgr.system_security_level < g_TRACE_mgr->ald_mgr.ald_resident_level)
					{
						(void)TRACE_ALD_interface_handle_ald_event(g_TRACE_mgr, &(g_TRACE_mgr->ald_mgr),FALSE);
					}

					while(exit_loop == FALSE)
					{
						nfds = epoll_wait(g_TRACE_mgr->ald_mgr.epollHandle, g_TRACE_mgr->ald_mgr.events,TRACE_ALD_MAX_EPOLL_EVENTS, -1);

						if(nfds < 0)
						{
							err_tmp = errno;
							if(err_tmp != EINTR)
							{
								TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "epoll_wait() failed in main !\n");
								exit_loop = TRUE;
							}
						}

						if(nfds > 0) /* epoll_wait() returned a proper handle */
						{
							for(i = 0; i < nfds; i++)
							{

								if(g_TRACE_mgr->ald_mgr.events[i].data.fd == g_TRACE_mgr->ald_mgr.file_desc)
								{
									rc = TRACE_ALD_interface_handle_ald_event(g_TRACE_mgr, &(g_TRACE_mgr->ald_mgr),TRUE);
									if(rc == E_OK)
									{
										TRACE_SYSLOG(TRACE_SYSLOG_NOTICE ,"ALD plugin message sent invoked Successfully \n");
									}
									else
									{
										TRACE_SYSLOG(TRACE_SYSLOG_ERROR ,"ALD plugin message sent invoked ERROR\n");
									}
								}
								else
								{
									//reset the eventfd
									if(read(event_fd,&tmp,sizeof(uint64_t))==0)
									{
										// WHAT TO DO HERE
									}

									/*TRACE_signal_hdlr posted event. There shutdown gracefully */
									exit_loop = TRUE;
									TRACE_SYSLOG(TRACE_SYSLOG_NOTICE ,"TRACE_signal_hdlr posted event. Trace_BE Shutting down gracefully\n");
								}
							}
						}
					} // End of while loop
#endif
			
					/* do clean-up */
					(void)TRACE_dlt_stop(g_TRACE_mgr);
				}
#ifndef TRACE_ALD_INTERFACE_ENABLED
            sem_destroy(&g_semExit);
#else
            if(event_fd >= 0)
            {
                close(event_fd);
            }

            if(NULL != g_TRACE_mgr)
            {
                rc = TRACE_ALD_interface_deinit_ald_plugin(&g_TRACE_mgr,&(g_TRACE_mgr->ald_mgr));
            }
#endif
			/* uninitialise Trace mgr and release shared memory */
			if(NULL != g_TRACE_mgr)
			{
				(void)TRACE_stop(&g_TRACE_mgr);
			}
			}
		}
	}
	return rc;/*agv2kor:- if error happens return error value instead of E_OK */
}

